// -*-c++-*-

/*!
  \file neck_scan_field.cpp
  \brief scan field with neck evenly
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "neck_scan_field.h"

#include "basic_actions.h"

#include <rcsc/player/player_agent.h>
#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/rect_2d.h>

#include <numeric>
#include <deque>

//#define DEBUG

namespace rcsc {

const double Neck_ScanField::INVALID_ANGLE = -360.0;

/*-------------------------------------------------------------------*/
/*!
  scan field by turn_neck
*/
bool
Neck_ScanField::execute( PlayerAgent * agent )
{
    static GameTime s_last_calc_time( 0, 0 );
    static ViewWidth s_last_calc_view_width = ViewWidth::NORMAL;
    static AngleDeg s_cached_target_angle( 0.0 );

    if ( s_last_calc_time != agent->world().time()
         || s_last_calc_view_width != agent->effector().queuedNextViewWidth() )
    {
        s_last_calc_time = agent->world().time();
        s_last_calc_view_width = agent->effector().queuedNextViewWidth();

        s_cached_target_angle = calcAngle( agent );
    }

    return agent->doTurnNeck( s_cached_target_angle
                              - agent->effector().queuedNextMyBody()
                              - agent->world().self().neck() );
}

/*-------------------------------------------------------------------*/
/*!

*/
AngleDeg
Neck_ScanField::calcAngle( const PlayerAgent * agent )
{
    double angle_for_wide = calcAngleForWidePitchEdge( agent );
    if ( angle_for_wide != INVALID_ANGLE )
    {
        return AngleDeg( angle_for_wide );
    }

    angle_for_wide = calcAnglePlayerCount( agent );
    if ( angle_for_wide != INVALID_ANGLE )
    {
        return AngleDeg( angle_for_wide );
    }

    return calcAngleDefault( agent );
}

/*-------------------------------------------------------------------*/
/*!

*/
AngleDeg
Neck_ScanField::calcAngleDefault( const PlayerAgent * agent )
{
    static const
        Rect2D pitch_rect( Vector2D( -ServerParam::i().pitchHalfLength(),
                                     -ServerParam::i().pitchHalfWidth() ),
                           Size2D( ServerParam::i().pitchLength(),
                                   ServerParam::i().pitchWidth() ) );
    static const
        Rect2D expand_pitch_rect( Vector2D( -ServerParam::i().pitchHalfLength() - 3.0,
                                            -ServerParam::i().pitchHalfWidth() - 3.0 ),
                                  Size2D( ServerParam::i().pitchLength() + 6.0,
                                          ServerParam::i().pitchWidth() + 6.0 ) );
    static const
        Rect2D goalie_rect( Vector2D( ServerParam::i().pitchHalfLength() - 3.0,
                                      -15.0 ),
                            Size2D( 10.0, 30.0 ) );

    const WorldModel & wm = agent->world();

    const double next_view_width = agent->effector().queuedNextViewWidth().width();

    const AngleDeg left_start
        = agent->effector().queuedNextMyBody()
        + ( ServerParam::i().minNeckAngle() - ( next_view_width * 0.5 ) );
    const double scan_range
        = ( ( ServerParam::i().maxNeckAngle() - ServerParam::i().minNeckAngle() )
            + next_view_width );

    dlog.addText( Logger::ACTION,
                  __FILE__": calcAngle() next_left_limit=%.0f, next_neck_range=%.0f",
                  left_start.degree(), scan_range );

    const double shrinked_scan_range = scan_range - WorldModel::DIR_STEP * 1.5;
    const double shrinked_next_view_width = next_view_width - WorldModel::DIR_STEP * 1.5;

    AngleDeg sol_angle = left_start + scan_range * 0.5;

    if ( scan_range < shrinked_next_view_width )
    {
        dlog.addText( Logger::ACTION,
                      __FILE__": calcAngleDefault() scan reange is smaller than next view width." );
        return sol_angle;
    }


    AngleDeg tmp_angle = left_start;

    const std::size_t size_of_view_width
        = static_cast< std::size_t >
        ( rint( shrinked_next_view_width / WorldModel::DIR_STEP ) );

    std::deque< int > dir_counts( size_of_view_width );

    // generate first visible cone score list
    {

        const std::deque< int >::iterator end = dir_counts.end();
        for ( std::deque< int >::iterator it = dir_counts.begin();
              it != end;
              ++it )
        {
            *it = wm.dirCount( tmp_angle );
            tmp_angle += WorldModel::DIR_STEP;
        }
    }

    int max_count_sum = 0;
    double add_dir = shrinked_next_view_width;

    dlog.addText( Logger::ACTION,
                  __FILE__": calcAngleDefault() loop start. left_start=%.0f shrinked_can_range=%.0f",
                  left_start.degree(),
                  shrinked_scan_range );

    const Vector2D my_next = agent->effector().queuedNextMyPos();
    const bool consider_pitch = ( wm.gameMode().type() == GameMode::PlayOn
                                  || ( wm.gameMode().type() != GameMode::IndFreeKick_
                                       && wm.gameMode().type() != GameMode::BackPass_
                                       && wm.ball().distFromSelf() < wm.self().playerType().playerSize() + 0.15 )
                                  );

    do
    {
        int tmp_count_sum = std::accumulate( dir_counts.begin(), dir_counts.end(), 0 );

        AngleDeg angle = tmp_angle - shrinked_next_view_width * 0.5;
#ifdef DEBUG
        dlog.addText( Logger::ACTION,
                      "___ angle=%.0f tmp_count_sum=%d",
                      angle.degree(),
                      tmp_count_sum );
#endif
        if ( tmp_count_sum > max_count_sum )
        {
            bool update = true;
            if ( consider_pitch )
            {
                {
                    Vector2D face_point
                        = my_next
                        + Vector2D::polar2vector( 20.0, angle );
                    if ( ! pitch_rect.contains( face_point )
                         && ! goalie_rect.contains( face_point ) )
                    {
                        update = false;
                    }
                }

                if ( update )
                {
                    Vector2D left_face_point
                        = my_next
                        + Vector2D::polar2vector( 20.0, angle - next_view_width*0.5 );
                    if ( ! expand_pitch_rect.contains( left_face_point )
                         && ! goalie_rect.contains( left_face_point ) )
                    {
                        update = false;
                    }
                }

                if ( update )
                {
                    Vector2D right_face_point
                        = my_next
                        + Vector2D::polar2vector( 20.0, angle + next_view_width*0.5 );
                    if ( ! expand_pitch_rect.contains( right_face_point )
                         && ! goalie_rect.contains( right_face_point ) )
                    {
                        update = false;
                    }
                }
            }

            if ( update )
            {
#ifdef DEBUG
                dlog.addText( Logger::ACTION,
                              "___-- updated" );
#endif
                sol_angle = angle;
                max_count_sum = tmp_count_sum;
            }
        }

        dir_counts.pop_front();
        add_dir += WorldModel::DIR_STEP;
        tmp_angle += WorldModel::DIR_STEP;
        dir_counts.push_back( wm.dirCount( tmp_angle ) );
    }
    while ( add_dir <= scan_range );


    dlog.addText( Logger::ACTION,
                  __FILE__": calcAngleDefault() target angle = %.0f",
                  sol_angle.degree() );

    return sol_angle;
}

/*-------------------------------------------------------------------*/
/*!

*/
double
Neck_ScanField::calcAngleForWidePitchEdge( const PlayerAgent * agent )
{
    if ( agent->effector().queuedNextViewWidth().type() != ViewWidth::WIDE )
    {
        return INVALID_ANGLE;
    }

    const WorldModel & wm = agent->world();

    if ( wm.gameMode().type() != GameMode::PlayOn
         && wm.gameMode().type() != GameMode::GoalKick_
         && wm.ball().distFromSelf() > 2.0 )
    {
        return INVALID_ANGLE;
    }

    const ServerParam & param = ServerParam::i();

    const Vector2D next_self_pos = wm.self().pos() + wm.self().vel();
    const double pitch_x_thr = param.pitchHalfLength() - 15.0;
    const double pitch_y_thr = param.pitchHalfLength() - 10.0;

    double target_angle = INVALID_ANGLE;

    if ( next_self_pos.absY() > pitch_y_thr )
    {
        Vector2D target_pos( param.pitchHalfLength() - 7.0, 0.0 );
        target_pos.x = std::min( target_pos.x, target_pos.x * 0.7 * next_self_pos.x * 0.3 );

        if ( next_self_pos.y > + pitch_y_thr )
        {
            target_angle = ( target_pos - next_self_pos ).th().degree();
            dlog.addText( Logger::ACTION,
                          __FILE__": calcAngleForWidePitchEdge() y+ point=(%.1f %.1f) dir=%.1f",
                          target_pos.x, target_pos.y,
                          target_angle );
        }

        if ( next_self_pos.y < - pitch_y_thr )
        {
            target_angle = ( target_pos - next_self_pos ).th().degree();
            dlog.addText( Logger::ACTION,
                          __FILE__": calcAngleForWidePitchEdge() y- point=(%.1f %.1f) dir=%.1f",
                          target_pos.x, target_pos.y,
                          target_angle );
        }
    }

    if ( next_self_pos.absX() > pitch_x_thr )
    {
        Vector2D target_pos( param.pitchHalfLength() * 0.5, 0.0 );

        if ( next_self_pos.x > + pitch_x_thr )
        {
            target_angle = ( target_pos - next_self_pos ).th().degree();
            dlog.addText( Logger::ACTION,
                          __FILE__": calcAngleForWidePitchEdge() x+ point=(%.1f %.1f) dir=%.1f",
                          target_pos.x, target_pos.y,
                          target_angle );
        }

        if ( next_self_pos.x < - pitch_x_thr )
        {
            target_angle = ( target_pos - next_self_pos ).th().degree();
            dlog.addText( Logger::ACTION,
                          __FILE__": calcAngleForWidePitchEdge() x- point=(%.1f %.1f) dir=%.1f",
                          target_pos.x, target_pos.y,
                          target_angle );
        }
    }

    if ( target_angle == INVALID_ANGLE )
    {
        dlog.addText( Logger::ACTION,
                      __FILE__": calcAngleForWidePitchEdge() no target" );
    }

    return target_angle;
}

/*-------------------------------------------------------------------*/
/*!

*/
double
Neck_ScanField::calcAnglePlayerCount( const PlayerAgent * agent )
{
    const double view_width = agent->effector().queuedNextViewWidth().width();
    if ( view_width < 90.0 )
    {
        return INVALID_ANGLE;
    }

    const WorldModel & wm = agent->world();

    if ( wm.gameMode().type() != GameMode::PlayOn
         && wm.gameMode().type() != GameMode::GoalKick_
         && wm.ball().distFromSelf() > 2.0 )
    {
        return INVALID_ANGLE;
    }

    if ( wm.allPlayers().size() <= 21 )
    {
        dlog.addText( Logger::ACTION,
                      __FILE__": calcAnglePlayerCount() no enough number of players %d",
                      wm.allPlayers().size() );
        return INVALID_ANGLE;
    }

//     if ( wm.gameMode().type() != GameMode::PlayOn
//          && wm.gameMode().type() != GameMode::GoalKick_ )
//     {
//         return INVALID_ANGLE;
//     }

    const ServerParam & param = ServerParam::i();

    const Vector2D next_self_pos = agent->effector().queuedNextMyPos();
    const AngleDeg next_self_body = agent->effector().queuedNextMyBody();
    const double view_half_width = view_width * 0.5;
    const double neck_step = ( param.maxNeckAngle() - param.minNeckAngle() ) / 36.0;
    const double neck_min = param.minNeckAngle();
    const double neck_max = param.maxNeckAngle() + 0.001;

    dlog.addText( Logger::ACTION,
                  __FILE__": calcAnglePlayerCount() next_body=%.1f  neck_step=%.1f",
                  next_self_body.degree(),
                  neck_step );

    double best_dir = INVALID_ANGLE;
    int best_score = 0;
    double best_angle_diff = 0.0;

    const AbstractPlayerCont::const_iterator end = wm.allPlayers().end();
    for ( double dir = neck_min; dir < neck_max; dir += neck_step )
    {
        const AngleDeg left_angle = next_self_body + ( dir -  view_half_width );
        const AngleDeg right_angle = next_self_body + ( dir +  view_half_width );

        int score = 0;
        double angle_diff = 180.0;

        for ( AbstractPlayerCont::const_iterator p = wm.allPlayers().begin();
              p != end;
              ++p )
        {
            if ( (*p)->isSelf() ) continue;
            //if ( (*p)->isGhost() ) continue;
            Vector2D pos = (*p)->pos() + (*p)->vel();
            if ( pos.dist( next_self_pos ) > 35.0 ) continue; // XXX magic number XXX

            AngleDeg angle = ( pos - next_self_pos ).th();
            if ( angle.isRightOf( left_angle )
                 && angle.isLeftOf( right_angle ) )
            {
                if ( (*p)->goalie()
                     && next_self_pos.x > 33.0 )
                {
                    score += (*p)->posCount() * 2 + 20;
                }
                else if ( (*p)->posCount() >= 10 )
                {
                    score += (*p)->posCount() * 2;;
                }
                else if ( pos.x > next_self_pos.x - 5.0 )
                {
                    score += (*p)->posCount() * 2 + 1;
                }
                else
                {
                    score += (*p)->posCount();
                }

                double tmp_diff = std::min( ( angle - left_angle ).abs(),
                                            ( angle - right_angle ).abs() );
                if ( tmp_diff < angle_diff )
                {
                    angle_diff = tmp_diff;
                }
            }
        }
#ifdef DEBUG
        dlog.addText( Logger::ACTION,
                      "__ dir=%.0f  score=%d  angle_diff=%.1f",
                      dir,
                      score,
                      angle_diff );
#endif
        if ( score > best_score )
        {
            best_dir = dir;
            best_score = score;
            best_angle_diff = angle_diff;
#ifdef DEBUG
            dlog.addText( Logger::ACTION,
                          "__ update best dir (1)" );
#endif
        }
        else if ( score == best_score )
        {
            if ( best_angle_diff < angle_diff )
            {
                best_dir = dir;
                best_score = score;
                best_angle_diff = angle_diff;
#ifdef DEBUG
                dlog.addText( Logger::ACTION,
                              "__ update best dir (2)" );
#endif
            }
        }
    }

    if ( best_dir != INVALID_ANGLE )
    {
        AngleDeg angle = next_self_body + best_dir;
        dlog.addText( Logger::ACTION,
                      __FILE__": calcAnglePlayerCount()  best_rel_dir=%.0f  angle=%.0f",
                      best_dir, angle.degree() );
        best_dir = angle.degree();
    }

    return best_dir;
}

}
