/*******************************************************************
*
*  DESCRIPTION: class Prioritizer
*
*  AUTHOR:  Shanta Ramchandani
*
*  EMAIL: shantar@sympatico.ca
*
*  DATE: October 20, 2003
*
*******************************************************************/

/** include files **/
#include "prioritizer.h"
#include "message.h"    // class ExternalMessage, InternalMessage
#include "mainsimu.h"   // MainSimulator::Instance().getParameter( ... )
#include "distri.h"     // class Distribution


/** public functions **/

/*******************************************************************
* Function Name: Prioritizer
* Description: 
********************************************************************/
Prioritizer::Prioritizer( const string &name )
: Atomic( name )
, in( addInputPort( "in" ) )
, numBuf1( addInputPort ( "numBuf1" ) )
, numBuf2( addInputPort ( "numBuf2" ) )
, numBuf3( addInputPort ( "numBuf3" ) )
, out1( addOutputPort( "out1" ) )
, out2( addOutputPort( "out2" ) )
, out3( addOutputPort( "out3" ) )
, inspectorReady( addOutputPort( "inspectorReady" ) )

{

		dist = Distribution::create( MainSimulator::Instance().getParameter( description(), "distribution" ) );
		MASSERT( dist ) ;
		for ( register int i = 0; i < dist->varCount(); i++ )
		{
			string parameter( MainSimulator::Instance().getParameter( description(), dist->getVar( i ) ) ) ;
			dist->setVar( i, str2Value( parameter ) ) ;
		}

}

/*******************************************************************
* Function Name: initFunction
* Description: 
* Precondition:
********************************************************************/
Model &Prioritizer::initFunction()
{
	buf1Status = 0;
	buf2Status = 0;
	buf3Status = 0;
	
	partType = -1;
	start = 1;
	working = 0;

	holdIn(active, Time::Zero);	//want to start-up with signaling inspectorReady

	return *this ;
}

/*******************************************************************
* Function Name: externalFunction
* Description: If an input occurs from port 'in' then wait some time before
* 		   sending an output. 
*		   If an input occurs from any other port, then update the value given
*		   and check to see if a part can be outputed.
********************************************************************/
Model &Prioritizer::externalFunction( const ExternalMessage &msg )
{

																		
	if (msg.port() == numBuf1) {
																		
		buf1Status = (int)msg.value();	//msg.value() == part type
		if (working == 0) holdIn( active, Time::Zero) ; 
	}

	else if (msg.port() == numBuf2) {
																		
		buf2Status = (int)msg.value();
		if (working == 0) holdIn( active, Time::Zero) ; 
	}

	else if (msg.port() == numBuf3) {
																		
		buf3Status = (int)msg.value();
		if (working == 0) holdIn( active, Time::Zero) ; 
	}

	else if ((msg.port() == in) && (partType == -1)) {
		working = 1;
		partType = (int)msg.value();
		holdIn( active, Time( static_cast< float >( fabs( distribution().get() ) ) ) ) ; 
	}

	return *this;
}

/*******************************************************************
* Function Name: internalFunction
* Description: 
********************************************************************/
Model &Prioritizer::internalFunction( const InternalMessage & )
{
	passivate();
	return *this ;
}

/*******************************************************************
* Function Name: outputFunction
* Description: This is where the actual prioritization occurs...
*		   Outputs output1, output2 and output3 receive a part based on 
*		   priorities. Output 1 has the highest priority and output3 has 
*		   the lowest. If all three buffers are full then do nothing.
********************************************************************/
Model &Prioritizer::outputFunction( const InternalMessage &msg )
{

																		
	if (working == 0 && partType == -1 && ((buf1Status < 2.0) || (buf2Status < 2.0) || (buf3Status < 2.0)) || start == 1 ) {
		sendOutput(msg.time(), inspectorReady, 1);
		start = 0;
	} else {																		
		if (partType != -1 && working == 1) {
			if ((buf1Status == 2.0) && (buf2Status == 2.0) && (buf3Status == 2.0))	{
				;// Do not send anything;
			} else if ( (buf3Status < buf1Status) && (buf3Status < buf2Status) ) {
				sendOutput(msg.time(), out3, partType);
				sendOutput(msg.time(), inspectorReady, 1);
				partType = -1;
				working = 0;
			} else if (buf2Status < buf1Status) {
				sendOutput(msg.time(), out2, partType);
				sendOutput(msg.time(), inspectorReady, 1);
				partType = -1;
				working = 0;
			} else {											
				sendOutput(msg.time(), out1, partType);
				sendOutput(msg.time(), inspectorReady, 1);
				partType = -1;
				working = 0;
			}
		}
	}
  
	return *this ;
}
