/*******************************************************************
*
*  DESCRIPTION: Atomic Model TailDetection
*
*  AUTHOR: Guillaume Plouffe
*
*  EMAIL: mailto://gplou010@uottawa.ca
*
*  DATE: 04/03/2016
*
*******************************************************************/

/** include files **/
#include "TailDetection.h"           // base header
#include "message.h"       // InternalMessage ....
#include "distri.h"        // class Distribution
#include "mainsimu.h"      // class MainSimulator


/*******************************************************************
* Function Name: TailDetection
* Description: constructor
********************************************************************/
TailDetection::TailDetection( const string &name )
: Atomic( name )
, speedIn( addInputPort( "speedIn" ) )
, startIn( addInputPort( "startIn" ) )	// startIn contain the length information
, scanStop( addOutputPort( "scanStop" ) )
, done( addOutputPort( "done" ) )
, lengthOut( addOutputPort( "lengthOut" ) )

{
	distFromScanner = 100;
	distScannerFromBin = 200;
	speed = 20;	// In cm/s
	isScanOver = false;

	string distFromScan( MainSimulator::Instance().getParameter( description(), "distFromScanner" ) ) ;

	if( distFromScan != "" )
	{
		distFromScanner = str2Int(distFromScan) ;
		cout << "distFromScanner = " << distFromScanner << endl;
	}

	string distFromBin( MainSimulator::Instance().getParameter( description(), "distScannerFromBin" ) ) ;
	if( distFromScan != "" )
	{
		distScannerFromBin = str2Int(distFromBin) ;
		cout << "distScannerFromBin = " << distScannerFromBin << endl;
	}


	try
	{
		dist = Distribution::create( MainSimulator::Instance().getParameter( description(), "distribution" ) );

		MASSERT( dist );

		for ( register int i = 0 ; i < dist->varCount() ; i++ )
		{
			string parameter( MainSimulator::Instance().getParameter( description(), dist->getVar(i) ) ) ;
			dist->setVar( i, str2float( parameter ) ) ;
		}
	}
	catch( InvalidDistribution &e )
	{
		e.addText( "The model " + description() + " has distribution problems!" ) ;
		e.print(cerr);
		MTHROW( e ) ;
	}
	catch( MException &e )
	{
		MTHROW( e ) ;
	}

}

/*******************************************************************
* Function Name: initFunction
********************************************************************/
Model &TailDetection::initFunction()
{
	this-> passivate();
	return *this ;

}

/*******************************************************************
* Function Name: externalFunction
* Description: This method executes when an external event is received.
********************************************************************/
Model &TailDetection::externalFunction( const ExternalMessage &msg )
{

	if (this->state() == passive){
		if( msg.port() == startIn)
		{
			length = static_cast<float>( fabs( distribution().get() ) );	// Randomly generated product length
			holdIn(active, Time( (length + distFromScanner)/speed  )) ;	// Wait for the product's tail to be detected then wait for the tail to reach the scanner
		}
		else if (msg.port() == speedIn)
		{
			speed = static_cast < int > (msg.value());
		}
		else
		{
			//Ignore inputs
		}
	}
	else if (this->state() == active)
	{
		// Ignore input
	}
	
	return *this ;
}

/*******************************************************************
* Function Name: internalFunction
* Description: This method executes when the TA has expired, right after the outputFunction has finished.
* 			   The new state and TA should be set.
********************************************************************/
Model &TailDetection::internalFunction( const InternalMessage & )
{
	if (this->state() == passive)
	{
		this-> passivate();	// Set to passive when the product's head reach the scanner


	}
	else if (this->state() == active)
	{
		if (isScanOver)
			holdIn(active, Time( distScannerFromBin/speed )) ;	// Wait for the product's tail reach the bin
		else
			this-> passivate();	// Set to passive when the product's head reach the scanner
	}


	return *this ;
}

/*******************************************************************
* Function Name: outputFunction
* Description: This method executes when the TA has expired. After this method the internalFunction is called.
*              Output values can be send through output ports
********************************************************************/
Model &TailDetection::outputFunction( const InternalMessage &msg )
{
	if (this->state() == active)
	{
		if (isScanOver){
			sendOutput(	msg.time(), done,  1) ;		// Warn the loader that the product inspection is over
			isScanOver = false;						// Reset value since inspection is over
		}
		else{
			sendOutput(	msg.time(), scanStop, 1 ) ;	// Command the scanner to stop
			sendOutput(	msg.time(), lengthOut, length ) ;
			isScanOver = true;
		}
	}
	return *this;

}

TailDetection::~TailDetection()
{
	//TODO: add destruction code here. Free distribution memory, etc. 
}
