/*******************************************************************
*
*  DESCRIPTION: Atomic Model Loader
*
*  AUTHOR: Guillaume Plouffe
*
*  EMAIL: mailto://gplou010@uottawa.ca
*
*  DATE: 04/03/2016
*
*******************************************************************/

/** include files **/
#include "LoaderType.h"           // base header
#include "message.h"       // InternalMessage ....
#include "distri.h"        // class Distribution
#include "mainsimu.h"      // class MainSimulator


/*******************************************************************
* Function Name: LoaderType
* Description: constructor
********************************************************************/
LoaderType::LoaderType( const string &name )
: Atomic( name )
, speedIn( addInputPort( "speedIn" ) )
, next( addInputPort( "next" ) )
, done( addInputPort( "done" ) )
, speedOut( addOutputPort( "speedOut" ) )
, start( addOutputPort( "start" ) )

// Initialization of variables
{
	readyToSend = 0;	// Number of product left in the loader
	speed = 20;		// In cm/s
}



/*******************************************************************
* Function Name: initFunction
********************************************************************/
Model &LoaderType::initFunction()
{
	this-> passivate();
	return *this ;
}

/*******************************************************************
* Function Name: externalFunction
* Description: This method executes when an external event is received.
********************************************************************/
Model &LoaderType::externalFunction( const ExternalMessage &msg )
{

	if (this->state() == passive){

		if( msg.port() == next)
		{
			readyToSend = static_cast < int > (msg.value()); // Number of product to inspect

			if (readyToSend >0)
			{
				holdIn(passive, Time::Zero) ;
			}


		}
		else if(msg.port() == speedIn)
		{
			speed = static_cast < int > (msg.value()); // speed of the conveyor
			if(speed<5){
				speed = 5;
			}
			else if (speed > 100)
			{
				speed = 100;
			}
		}
		else
		{
			//Ignore inputs
		}
	}
	else if (this->state() == active)
	{
		if (msg.port() == done){

			holdIn(active, Time::Zero) ;	// Transit immediatly to next state;
		}

	}
	
	return *this ;
}

/*******************************************************************
* Function Name: internalFunction
* Description: This method executes when the TA has expired, right after the outputFunction has finished.
* 			   The new state and TA should be set.
********************************************************************/
Model &LoaderType::internalFunction( const InternalMessage & )
{
	if(readyToSend>0){
		readyToSend--;
		holdIn(active, Time::Inf) ;
	}
	else{
		  this-> passivate();
	}

	return *this ;
}

/*******************************************************************
* Function Name: outputFunction
* Description: This method executes when the TA has expired. After this method the internalFunction is called.
*              Output values can be send through output ports
********************************************************************/
Model &LoaderType::outputFunction( const InternalMessage &msg )
{
	
	if (readyToSend>0)
		{
			sendOutput(	msg.time(), speedOut, speed ) ;
			sendOutput(	msg.time(), start, 1 ) ;
		}


	return *this;

}

LoaderType::~LoaderType()
{
	//TODO: add destruction code here. Free distribution memory, etc. 
}
