/*******************************************************************
*
*  DESCRIPTION: Atomic Model Floor2
*
*  AUTHOR: Sixuan Wang
*
*  EMAIL: mailto://swang@sce.carleton.ca
*
*  DATE: 15/10/201S2
*
*******************************************************************/

/** include files **/
#include "floor2.h"      // class Floor
#include "message.h"    // class ExternalMessage, InternalMessage
#include "mainsimu.h"   // MainSimulator::Instance().getParameter( ... )
#include  <stdlib.h>
#include "distri.h"        // class Distribution
#include "strutil.h"       // str2float( ... )

/** public functions **/

/*******************************************************************
* Function Name: Floor2
* Description: 
********************************************************************/
Floor2::Floor2( const string &name )
: Atomic( name )
, in( addOutputPort( "in" ) )
, out( addOutputPort( "out" ) )
, capaOut( addOutputPort( "capaOut" ) )
, occuOut2( addOutputPort( "occuOut2" ) )
, visitingQueueSize (10)
, currentDuration( 0, 0, 0, 0)
{
	string size( MainSimulator::Instance().getParameter( description(), "queueSize" ) ) ;
	if( size != "")
		visitingQueueSize = atoi(size.c_str());

	try
		{
			dist = Distribution::create( MainSimulator::Instance().getParameter( description(), "distribution" ) );

			MASSERT( dist );

			for ( register int i = 0 ; i < dist->varCount() ; i++ )
			{
				string parameter( MainSimulator::Instance().getParameter( description(), dist->getVar(i) ) ) ;
				dist->setVar( i, str2float( parameter ) ) ;
				distribution().get();
				distribution().get();
			}
		} catch( InvalidDistribution &e )
		{
			e.addText( "The model " + description() + " has distribution problems!" ) ;
			e.print(cerr);
			MTHROW( e ) ;
		} catch( MException &e )
		{
			MTHROW( e ) ;
		}
}

/*******************************************************************
* Function Name: initFunction
* Description: Resetea la lista
* Precondition: El tiempo del proximo evento interno es Infinito
********************************************************************/
Model &Floor2::initFunction()
{
	receivable = true;
	receivableLast = true;
	occupation = 0.0;
	this-> passivate();
	return *this ;
}

/*******************************************************************
* Function Name: externalFunction
* Description: 
********************************************************************/
Model &Floor2::externalFunction( const ExternalMessage &msg )
{
	if( msg.port() == in && !isQueueFull())
	{
		int id = static_cast < int > (msg.value());
		Time t1 (msg.time());
		Time t2 (static_cast<float>( fabs(distribution().get()) ));
		Time t = t1 + t2;

		visitingQueue.insert(pair<int,Time>(id, t));

		if(state()==passive) { //first person, run internal directly
			currentPID = id;
			currentDuration = t;
			holdIn(active, t2);
		}
		else if(isQueueFull()) { //full, set receivable and inform Gate to stop
			receivable = false;
			holdIn(active, Time::Zero);
		}
	}
	return *this;
}

/*******************************************************************
* Function Name: internalFunction
* Description: 
********************************************************************/
Model &Floor2::internalFunction( const InternalMessage &msg )
{
	if(!receivable) {
		if(currentDuration > msg.time())  //current one doesn't finish
			holdIn(active, currentDuration - msg.time());
		else { //current one finish
			receivable = true; //go to following
			holdIn(active, Time::Zero);
		}
	}else if(!isQueueEmpty()) {//queue is not empty
		map<int ,Time>::iterator iterOld = visitingQueue.find(currentPID); //find the smallest
		visitingQueue.erase(iterOld);  //remove from map

		if(!isQueueEmpty()) {//queue is still not empty
			currentPID = getSmallestDuePerson();
			map<int ,Time>::iterator iter = visitingQueue.find(currentPID); //find the smallest
			currentDuration = iter->second;
			if(currentDuration > msg.time())
				holdIn(active, iter->second - msg.time());  //remaining time
			else holdIn(active, Time::Zero);  //some one is waiting to send, pick him
		} else {
			this-> passivate();
		}
	}
	else
		this-> passivate();
	return *this ;
}

/*******************************************************************
* Function Name: outputFunction
* Description: 
********************************************************************/
Model &Floor2::outputFunction( const InternalMessage &msg )
{
	if(!receivable) {
		if(receivableLast) {
			sendOutput(msg.time(), capaOut, 0);
			receivableLast = false;
		}
	} else if(!isQueueEmpty()) {//queue is not empty
		//calculate the time when erasing (just before erasing in internalFun)
//		float f1 = (msg.time()-lastOccuCalTime).asMsecs() * visitingQueue.size();  //weight from lastOccuCalTime to current
//		float f2 = lastOccuCalTime.asMsecs() * occupation*visitingQueueSize;   //weight since lastOccuCalTime
//		occupation = (f1 + f2)/(msg.time().asMsecs() * visitingQueueSize);
//		lastOccuCalTime = msg.time();

		occupation =  (float)visitingQueue.size()/(float)visitingQueueSize;
		sendOutput(msg.time(), occuOut2, occupation);

		sendOutput(msg.time(), out, currentPID);
		if(!receivableLast) {
			sendOutput(msg.time(), capaOut, 1);
			receivableLast = true;
		}
	}

	return *this ;
}

/*******************************************************************
* Function Name: isQueueEmpty
* Description:
********************************************************************/
bool Floor2::isQueueEmpty() {
	if(visitingQueue.size()==0) return true;
	else return false;
}

/*******************************************************************
* Function Name: isQueueFull
* Description:
********************************************************************/
bool Floor2::isQueueFull() {
	if(visitingQueue.size()==visitingQueueSize) return true;
	else return false;

}

/*******************************************************************
* Function Name: outputFunction
* Description:
********************************************************************/
int Floor2::getSmallestDuePerson() {
	Time tmp1, tmp2;
	int result = 0;

	map <int, Time>::iterator iter;
	for (iter = visitingQueue.begin(); iter != visitingQueue.end(); iter++) {
		if(iter == visitingQueue.begin()){ //first one
			tmp1 = iter->second;
			result = iter->first;
		} else {
			tmp2 = iter->second;
			if(tmp2 < tmp1){
				tmp1 = tmp2;  //find smallest one
				result = iter->first;
			}
		}
	}
	return result;
}

Floor2::~Floor2()
{
	delete dist;
}


