/*******************************************************************
*
*  DESCRIPTION: Atomic Model Exit
*
*  AUTHOR: Sixuan Wang
*
*  EMAIL: mailto://swang@sce.carleton.ca
*
*  DATE: 15/10/201S2
*
*******************************************************************/

/** include files **/
#include "exit.h"      // class Queue
#include "message.h"    // class ExternalMessage, InternalMessage
#include "mainsimu.h"   // MainSimulator::Instance().getParameter( ... )

/** public functions **/

/*******************************************************************
* Function Name: Exit
* Description: 
********************************************************************/
Exit::Exit( const string &name )
: Atomic( name )
, in( addOutputPort( "in" ) )
, occuIn1( addOutputPort( "occuIn1" ) )
, occuIn2( addOutputPort( "occuIn2" ) )
, throughPut( addOutputPort( "throughPut" ) )
, occuOut1( addOutputPort( "occuOut1" ) )
, occuOut2( addOutputPort( "occuOut2" ) )
, occu1LastTime( 0, 0, 0, 0)
, occu2LastTime( 0, 0, 0, 0)
, preparationTime( 0, 0, 0, 0)
{
	string time( MainSimulator::Instance().getParameter( description(), "preparation" ) ) ;

		if( time != "" )
			preparationTime = time ;
}

/*******************************************************************
* Function Name: initFunction
* Description: Resetea la lista
* Precondition: El tiempo del proximo evento interno es Infinito
********************************************************************/
Model &Exit::initFunction()
{
	throughPutV = 0;
	occupation1 = 0.0;
	occupation2 = 0.0;
	tpOut = false;
	o1Put = false;
	o2Put = false;
	this-> passivate();
	return *this ;
}

/*******************************************************************
* Function Name: externalFunction
* Description: 
********************************************************************/
Model &Exit::externalFunction( const ExternalMessage &msg )
{
	if( msg.port() == in )
	{
		throughPutV++;

		tpOut = true;
		holdIn(active, preparationTime);
	} else if( msg.port() == occuIn1 )
	{
		float currentOccu = static_cast < float > (msg.value());
		float f2 = occupation1 * occu1LastTime.asMsecs();   //weight since lastOccuCalTime
		float f1 = currentOccu * (msg.time()-occu1LastTime).asMsecs();  //weight from lastOccuCalTime to current

		occupation1 = (float)(f2 + f1)/(float)msg.time().asMsecs();
		occu1LastTime = msg.time();

		o1Put = true;
		holdIn(active, Time::Zero);
	} else if( msg.port() == occuIn2 )
	{
		float currentOccu = static_cast < float > (msg.value());
		float f2 = occupation2 * occu2LastTime.asMsecs();   //weight since lastOccuCalTime
		float f1 = currentOccu * (msg.time()-occu2LastTime).asMsecs();  //weight from lastOccuCalTime to current

		occupation2 = (float)(f2 + f1)/(float)msg.time().asMsecs();
		occu2LastTime = msg.time();

		o2Put = true;
		holdIn(active, Time::Zero);
	}

	return *this;
}

/*******************************************************************
* Function Name: internalFunction
* Description: 
********************************************************************/
Model &Exit::internalFunction( const InternalMessage & )
{
	this-> passivate();
	return *this ;
}

/*******************************************************************
* Function Name: outputFunction
* Description: 
********************************************************************/
Model &Exit::outputFunction( const InternalMessage &msg )
{
	if(tpOut) {
		sendOutput(msg.time(), throughPut, throughPutV);
		tpOut = false;
	}
	if(o1Put) {
		sendOutput(msg.time(), occuOut1, occupation1);
		o1Put = false;
	}
	if(o2Put) {
		sendOutput(msg.time(), occuOut2, occupation2);
		o2Put = false;
	}
	return *this ;
}
