/*******************************************************************
*
*  DESCRIPTION: Atomic Model SecondaryDecider
*
*  AUTHOR: Scott Stewart
*
*  EMAIL: wtime@live.ca
*
*  DATE: 4/3/2016
*
*******************************************************************/
/** include files **/
#include "secondaryDecider.h"      		// class Line
#include "distri.h"        // class Distribution
#include "message.h"        // class ExternalMessage, InternalMessage
#include "mainsimu.h"       // MainSimulator::Instance().getParameter( ... )
#include "strutil.h"       // str2float( ... )
#include <stdlib.h>

/** public functions **/

/*******************************************************************
* Function Name:
* Description:
********************************************************************/
SecondaryDecider::SecondaryDecider( const string &name )
: Atomic( name )
, next_person( addInputPort( "next_person" ) )
, ok( addOutputPort( "ok" ) )
, turn_away( addOutputPort( "turn_away" ) )
, is_free( addOutputPort( "is_free" ) )
{
	id = 1;
	try
	{
		dist = Distribution::create( MainSimulator::Instance().getParameter( description(), "distribution" ) );

		MASSERT( dist );

		for ( register int i = 0 ; i < dist->varCount() ; i++ )
		{
			string parameter( MainSimulator::Instance().getParameter( description(), dist->getVar(i) ) ) ;
			dist->setVar( i, str2float( parameter ) ) ;
		}
	} catch( InvalidDistribution &e )
	{
		e.addText( "The model " + description() + " has distribution problems!" ) ;
		e.print(cerr);
		MTHROW( e ) ;
	} catch( MException &e )
	{
		MTHROW( e ) ;
	}
}

/*******************************************************************
* Function Name: initFunction
* Description: Resets the list
* Precondition: The time of the next internal event is infinite
********************************************************************/
Model &SecondaryDecider::initFunction()
{
	passivate();
	return *this ;
}

/*******************************************************************
* Function Name: externalFunction
* Description:
********************************************************************/
Model &SecondaryDecider::externalFunction( const ExternalMessage &msg )
{
	if( msg.port() == next_person ) //New person is sent to this decider
	{
		id = static_cast < int > (msg.value());
		cout << id << "\n";
		holdIn( active, Time( static_cast<float>( fabs( distribution().get() ) ) ) );
	}

	return *this;
}

/*******************************************************************
* Function Name: internalFunction
* Description:
********************************************************************/
Model &SecondaryDecider::internalFunction( const InternalMessage & )
{
    passivate(); //When the decider has decided on the person, the decider will wait for the next person in line.
	return *this ;
}

/*******************************************************************
* Function Name: outputFunction
* Description:
********************************************************************/
Model &SecondaryDecider::outputFunction( const InternalMessage &msg )
{
	if ((double)rand() / (double) RAND_MAX  < 0.50) // 50% chance it will be ok.
	{
		sendOutput( msg.time(), ok, id ) ;
	}else //Otherwise, it will be turned away
	{
		sendOutput( msg.time(), turn_away, id ) ;
	}

	sendOutput( msg.time(), is_free, id ) ; //Whatever happens, the decider will be marked as free for the line.

	return *this ;
}
